/**
 * External dependencies
 */
import classnames from 'classnames';
import memize from 'memize';

/**
 * WordPress dependencies
 */
import { __, sprintf } from '@wordpress/i18n';
import { dateI18n } from '@wordpress/date';
import { Button, Modal } from '@wordpress/components';
import { compose, withState, withInstanceId } from '@wordpress/compose';

/**
 * Internal dependencies
 */
import { getConfigValue } from '../../utils';
import ScanOnly from './scan-only';
import Header, { Title, Date } from '../../components/card/header';
import { FooterSchemaActions } from '../../components/card/footer';
import { MalwareScanResults, SiteScanResults, PrintR } from '@ithemes/security-components';
import './style.scss';

function getScanStatus( entry ) {
	switch ( entry.code ) {
		case 'scan-failure-server-error':
		case 'scan-failure-client-error':
		case 'sucuri-system-error':
		case 'error':
			return [ 'warn', __( 'Scan Error', 'it-l10n-ithemes-security-pro' ) ];
		case 'clean':
			return [ 'clean', __( 'Clean', 'it-l10n-ithemes-security-pro' ) ];
		default:
			if ( entry.code.includes( '--' ) ) {
				const parts = entry.code.split( '--' ).map( ( part ) => {
					switch ( part ) {
						case 'found-malware':
							return __( 'Found Malware', 'it-l10n-ithemes-security-pro' );
						case 'on-blacklist':
							return __( 'On Blacklist', 'it-l10n-ithemes-security-pro' );
						case 'vulnerable-software':
							return __( 'Vulnerable Software', 'it-l10n-ithemes-security-pro' );
						case 'has-error':
							return __( 'Scan Error', 'it-l10n-ithemes-security-pro' );
						default:
							return part;
					}
				} );

				return [ 'error', parts.join( ', ' ) ];
			}

			return [ 'unknown', entry.code ];
	}
}

const isSameUrl = memize( ( a, b ) => String( a ).replace( /https?:\/\//, '' ) === String( b ).replace( /https?:\/\//, '' ) );

function MalwareScan( { card, config, viewEntry, scanResults, showRawDetails, setState, instanceId } ) {
	return (
		<div className="itsec-card--type-malware-scan">
			<Header>
				<Title card={ card } config={ config } />
				<Date card={ card } config={ config } />
			</Header>
			<section className="itsec-card-malware-scan__scans-section">
				<table className="itsec-card-malware-scan__scans">
					<thead>
						<tr>
							<th>{ __( 'Time', 'it-l10n-ithemes-security-pro' ) }</th>
							<th>{ __( 'Status', 'it-l10n-ithemes-security-pro' ) }</th>
							<th><span className="screen-reader-text">{ __( 'Actions', 'it-l10n-ithemes-security-pro' ) }</span></th>
						</tr>
					</thead>
					<tbody>
						{ card.data.map( ( log ) => {
							const [ status, label ] = getScanStatus( log );

							return (
								<tr key={ log.id }>
									<th scope="row">{ dateI18n( 'M d, Y g:i A', log.time ) }</th>
									<td>
										<span className={ classnames( 'itsec-card-malware-scan__scan-status', `itsec-card-malware-scan__scan-status--${ status }` ) }>
											{ label }
										</span>
									</td>
									<td>
										<Button isLink aria-pressed={ viewEntry === log.id } onClick={ () => setState( { viewEntry: log.id } ) }>
											{ __( 'View', 'it-l10n-ithemes-security-pro' ) }
										</Button>
										{ viewEntry === log.id && (
											<Modal
												title={ sprintf( __( 'View Scan Details for %s', 'it-l10n-ithemes-security-pro' ), dateI18n( 'M d, Y g:i A', log.time ) ) }
												onRequestClose={ () => setState( { viewEntry: 0, showRawDetails: false } ) }
											>
												{ log.results.entries ?
													<SiteScanResults results={ log.results } showSiteUrl={ ! isSameUrl( log.results.url, getConfigValue( 'site_url' ) ) } /> :
													<MalwareScanResults results={ log.results } />
												}
												<Button className="itsec-card-malware-scan__raw-details-toggle" isLink onClick={ () => setState( { showRawDetails: ! showRawDetails } ) }
													aria-expanded={ showRawDetails } aria-controls={ `itsec-card-malware-scan__raw-details--${ instanceId }` }>
													{ showRawDetails ? __( 'Hide Raw Details', 'it-l10n-ithemes-security-pro' ) : __( 'Show Raw Details', 'it-l10n-ithemes-security-pro' ) }
												</Button>
												<div id={ `itsec-card-malware-scan__raw-details--${ instanceId }` } style={ { visibility: showRawDetails ? 'visible' : 'hidden' } }>
													{ showRawDetails && <PrintR json={ log.results } /> }
												</div>
											</Modal>
										) }
									</td>
								</tr>
							);
						} ) }
					</tbody>
				</table>
			</section>
			<FooterSchemaActions card={ card } onComplete={ ( href, response ) => href.endsWith( '/scan' ) && setState( { scanResults: response.results } ) } />
			{ scanResults && (
				<Modal
					title={ __( 'Scan Results', 'it-l10n-ithemes-security-pro' ) }
					onRequestClose={ () => setState( { scanResults: undefined } ) }
				>
					{ scanResults.entries ? <SiteScanResults results={ scanResults } showSiteUrl={ false } /> : <MalwareScanResults results={ scanResults } /> }
				</Modal>
			) }
		</div>
	);
}

export const slug = 'malware-scan';
export const settings = {
	render: getConfigValue( 'db_logs' ) ? compose( [
		withState( { viewEntry: 0, scanResults: undefined, showRawDetails: false } ),
		withInstanceId,
	] )( MalwareScan ) : ScanOnly,
};
