<?php

class ITSEC_Malware_Scheduling_Scanner {

	/**
	 * Perform the scheduled scan and send an email summarizing the results.
	 *
	 * @param int $site_id
	 *
	 * @return null|array|WP_Error
	 */
	public static function scan( $site_id = 0 ) {
		require_once( ITSEC_Core::get_core_dir() . '/modules/malware/class-itsec-malware-scanner.php' );

		$results = ITSEC_Malware_Scanner::scan( $site_id );

		if ( ITSEC_Malware_Scanner::is_sucuri_error( $results ) ) {
			$results->add( 'itsec-scheduled-malware-scanner-sucuri-error', __( 'Malware scanning is temporarily unavailable, please try again later.', 'it-l10n-ithemes-security-pro' ) );

			return $results;
		}

		if ( ITSEC_Lib_Remote_Messages::has_action( 'malware-scanner-disable-malware-warnings' ) ) {
			return $results;
		}

		self::send_email( $results );

		return $results;
	}

	public static function send_email( $results ) {

		$nc = ITSEC_Core::get_notification_center();

		if ( ! $nc->is_notification_enabled( 'malware-scheduling' ) ) {
			return;
		}

		if ( null !== apply_filters( 'itsec_pre_malware_scanner_send_scheduled_results', null, $results ) ) {
			return;
		}

		$subject = self::get_email_subject( $results );

		if ( empty( $subject ) ) {
			return;
		}

		$mail = $nc->mail();
		$mail->set_subject( $subject );
		$mail->set_recipients( $nc->get_recipients( 'malware-scheduling' ) );

		$mail->add_header(
			esc_html__( 'Malware Scan', 'it-l10n-ithemes-security-pro' ),
			sprintf( esc_html__( 'Malware Scan for %s', 'it-l10n-ithemes-security-pro' ), '<b>' . date_i18n( get_option( 'date_format' ) ) . '</b>' )
		);

		self::add_email_message( $mail, $results );
		$mail->add_footer();


		$nc->send( 'malware-scheduling', $mail );
	}

	protected static function add_email_message( ITSEC_Mail $mail, $results ) {
		$site_name = get_bloginfo( 'name', 'display' );

		if ( is_wp_error( $results ) ) {
			$mail->add_list( array(
				/* translators: 1. Site name. */
				sprintf( esc_html__( 'An error occurred while running the scheduled malware scan on %s:', 'it-l10n-ithemes-security-pro' ), $site_name ),
				sprintf( esc_html__( 'Error Message: %s', 'it-l10n-ithemes-security-pro' ), $results->get_error_message() ),
				sprintf( esc_html__( 'Error Code: %s', 'it-l10n-ithemes-security-pro' ), '<code>' . esc_html( $results->get_error_code() ) . '</code>' ),
			), true );

			return;
		}

		$has_malware  = ! empty( $results['MALWARE']['WARN'] );
		$other_issues = 0;

		if ( ! empty( $results['SYSTEM']['ERROR'] ) ) {
			$other_issues += count( $results['SYSTEM']['ERROR'] );
		}
		if ( ! empty( $results['BLACKLIST']['WARN'] ) ) {
			$other_issues += count( $results['BLACKLIST']['WARN'] );
		}

		if ( $has_malware && $other_issues ) {
			/* translators: 1. Opening bold, 2. Closing bold, 3. Site title */
			$message = sprintf( esc_html( _n(
				'Sucuri\'s malware scanner found %1$smalware%2$s and another issue when scanning %3$s.',
				'Sucuri\'s malware scanner found %1$smalware%2$s and other issues when scanning %3$s.',
				$other_issues,
				'it-l10n-ithemes-security-pro'
			) ), '<b>', '</b>', $site_name );
		} elseif ( $has_malware ) {
			/* translators: 1. Opening bold, 2. Closing bold, 3. Site title */
			$message = sprintf( esc_html__( 'Sucuri\'s malware scanner found %1$smalware%2$s when scanning %3$s.', 'it-l10n-ithemes-security-pro' ), '<b>', '</b>', $site_name );
		} else {
			/* translators: 1. Site title */
			$message = sprintf( esc_html( _n(
				'Sucuri\'s malware scanner found an issue when scanning %s.',
				'Sucuri\'s malware scanner found issues when scanning %s.',
				$other_issues,
				'it-l10n-ithemes-security-pro'
			) ), $site_name );
		}

		$mail->add_text( $message );

		if ( ! empty( $results['SYSTEM']['ERROR'] ) ) {
			$mail->add_section_heading( esc_html( _n( 'Possible Site Issue', 'Possible Site Issues', count( $results['SYSTEM']['ERROR'] ), 'it-l10n-ithemes-security-pro' ) ) );
			$mail->add_list( array_map( 'esc_html', (array) $results['SYSTEM']['ERROR'] ) );
		}

		if ( ! empty( $results['BLACKLIST']['WARN'] ) ) {
			$mail->add_section_heading( esc_html__( 'Site Blacklisted', 'it-l10n-ithemes-security-pro' ) );
			$list = array();

			foreach ( (array) $results['BLACKLIST']['WARN'] as $warning ) {
				/* translators: 1. Blacklisted by, 2. Opening link, 3. Closing link. */
				$list[] = sprintf( esc_html__( '%1$s (%2$sdetails%3$s)', 'it-l10n-ithemes-security-pro' ), esc_html( $warning[0] ), esc_url( $warning[1] ) );
			}

			$mail->add_list( $list );
		}

		$mail->add_button( esc_html__( 'View Report', 'it-l10n-ithemes-security-pro' ), self::get_logs_link() );
	}

	protected static function get_logs_link() {
		return ITSEC_Mail::filter_admin_page_url( ITSEC_Core::get_logs_page_url( 'malware' ) );
	}

	protected static function get_email_subject( $results ) {
		if ( is_wp_error( $results ) ) {
			return __( 'Scheduled malware scan resulted in an error', 'it-l10n-ithemes-security-pro' );
		} else if ( ! empty( $results['BLACKLIST']['WARN'] ) || ! empty( $results['MALWARE']['WARN'] ) ) {
			return __( 'Scheduled malware scan found warnings', 'it-l10n-ithemes-security-pro' );
		} else if ( ! empty( $results['SYSTEM']['ERROR'] ) ) {
			$first_error = is_array( $results['SYSTEM']['ERROR'] ) ? $results['SYSTEM']['ERROR'][0] : $results['SYSTEM']['ERROR'];

			if ( ! preg_match( '/Unable to connect\./i', $first_error ) ) {
				return __( 'Scheduled malware scan found an error', 'it-l10n-ithemes-security-pro' );
			}
		}

		return false;
	}
}
